#! /usr/bin/env python3

# $+HEADER$
#
# Copyright 2017-2018 Christoph Lueders
#
# This file is part of the PtCut project: <http://wrogn.com/ptcut>
#
# PtCut is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# PtCut is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with PtCut.  If not, see <http://www.gnu.org/licenses/>.
#
# $-HEADER$

from __future__ import print_function, division
try:
    from sage.all_cmdline import *    # import sage library
    IS_SAGE = True
except ImportError:
    IS_SAGE = False

vsn = "3.0.2"
chull_f1 = 2
chull_f2 = 4
chull_f3 = 0.5
chull_f4 = 2

import os
import sys
import time
import numpy as np
import itertools
from util import *
from biomd import tropicalize_system, load_satyas_solution, biomd_simple, biomd_fast, biomd_slow, biomd_slowhull, biomd_easy, biomd_all, biomd_hard, read_grid_data, sample_grid
from prt import prt
from phwrapper import *
from math import log, log10

if not IS_SAGE:
    from vector import vector


def generate_polyhedra(points, bagnb, verbose=0, complex=False):
    """
    From a dictionary of points with their sign (point is the key, sign the value)
    generate polyhedra.  The first coordinate of the keys is the absolute value,
    i.e. an entry of [1,7,3,4] represents the equality 1+7x1+3x2+4x3=0.

    Input: the input list of points.
        Example: points = {(0,6,0): -1, (0,3,1): 1, (0,3,0): -1, (0,1,2): 1}

    Each point is the tropicalization of a monomial.
    The value in the dictionary is +/-1 and contains the sign of that monomial.
    If the value is 0, then the opposite sign rule doesn't apply for that point.
    """

    vs = list(points.keys())
    # all points must have the same number of coordinates
    assert all([len(p) == len(vs[0]) for p in points])

    # build a polyhedron of all points
    newton = phwrap(vertices=vs)

    if verbose:
        prt("Points: {}".format(len(points)))
        prt("Dimension of the ambient space: {}".format(newton.space_dim()))
        prt("Dimension of the Newton polytope: {}".format(newton.dim()))

        prt(("Points on the border of the Newton polytope" + (" ({} total):" if verbose > 1 else ": {}")).format(newton.n_vertices()), flush=True)
        if verbose > 1:
            for i in newton.vertices():
                prt("    {}".format(i))
        #prt("Newton Hrep:")
        #prt("  eqns={}".format(newton.equalities_list()))
        #prt("  ieqs={}".format(newton.inequalities_list()))

    # cycle through all edges (1-faces) of the boundary of the Newton polytope
    l = []
    cnt = 0
    if verbose:
        prt("Border edges{}:".format("" if complex else " between points with opposing signs"), flush=True)
    for e in itertools.combinations(newton.vertices(), 2):
        v1 = vector(e[0])                               # one endpoint of the edge
        v2 = vector(e[1])                               # the other endpoint
        # must use tuple for dict access
        if (complex or points[e[0]] * points[e[1]] <= 0) and adjacent_vertices(newton, v1, v2):
            # complex or opposing sign condition is met
            d = v1 - v2                                 # edge connecting both vertices
            # build list of inequalities:
            # v_1 = v_2 <= v_i, hence v_i - v_1 >= 0
            ie = []
            for v in newton.vertices():
                vi = vector(v)
                if vi != v1 and vi != v2:
                    ie.append(vi - v1)
            # eqns - list of equalities. Each line can be specified as any iterable container of
            #     base_ring elements. An entry equal to [1,7,3,4] represents the equality 1+7x1+3x2+4x3=0
            # ieqs - list of inequalities. Each line can be specified as any iterable container of
            #     base_ring elements. An entry equal to [1,7,3,4] represents the inequality 1+7x1+3x2+4x3>=0
            p = phwrap(eqns=[d], ieqs=ie)
            if verbose:
                prt("    {}: dim={}, compact={}".format(p.Hrep(), p.dim(), p.is_compact()))
            if verbose > 1:
                prt("    eq={}, ieqs={}".format(d, ie))
            if not p.is_empty():                        # exclude empty polyhedra
                p.combo = {bagnb: cnt}
                cnt += 1
                l.append(p)
    if not l and len(points) == 1:
        prt("Warning: formula defines no polyhedron!  Ignored.")
        l = None
    if verbose:
        prt()
    sys.stdout.flush()
    return l


def adjacent_vertices(p, v, w):
    """
    Check if two vertices (v and w) are adjacent,
    i.e. share the same facet of a polyhedron p.

    >>> p = phwrap(vertices=[[-1, -1, -1], [-1, -1, 1], [-1, 1, -1], [-1, 1, 1], [1, -1, -1], [1, -1, 1], [1, 1, -1], [1, 1, 1]])
    >>> e = phwrap(eqns=[(1,2,4,6)])
    >>> q = p & e

    #>>> l = sorted(q.vertices_list())
    #>>> l
    #[[-1, -1, 5/6], [-1, 1, -1/2], [1/2, 1, -1], [1, -1, 1/6], [1, 3/4, -1]]
    >>> adjacent_vertices(q, vector([1, -1, 1/6]), vector([-1, -1, 5/6]))
    True
    >>> adjacent_vertices(q, vector([-1, 1, -1/2]), vector([1/2, 1, -1]))
    True
    >>> adjacent_vertices(q, vector([1, 3/4, -1]), vector([-1, -1, 5/6]))
    False
    """
    #return True
    if __debug__:
        # make sure we only get points that are on the surface of the polyhedron.
        assert v != w
        # all the equalities must be true for both.
        for i in p.equalities_list():
            ii = vector(i[1:])
            assert ii*v + i[0] == 0
            assert ii*w + i[0] == 0
        # all the inequalities must be true for both as well.
        for i in p.inequalities_list():
            ii = vector(i[1:])
            assert ii*v + i[0] >= 0
            assert ii*w + i[0] >= 0
    # at least one inequality must be shared as well.
    # check for equality, since we're only checking vertices on the hull anyway.
    cnt = 0
    for i in p.inequalities_list():
        ii = vector(i[1:])
        if ii*v + i[0] == 0 and ii*w + i[0] == 0:
            cnt += 1
    # since an edge has dimension 1, the two points have to share at least dim-1 facets.
    return cnt >= p.dim() - 1


class PtsBag(list):
    """
    A bag (set) of polyhedra.  Subclassed only so we can add another member.
    """
    def __init__(self, l):
        list.__init__(self, l)
        self.name = ""

def make_polyhedra(ts, skip_formulas=[], verbose=0, complex=False):
    """
    From the tropicalization, create the polyhedra per bag.
    """
    assert ts
    start = mytime()
    pts_bags = []
    bagnb = 0
    for cnt,t in enumerate(ts):
        prt(end="[{}]".format(cnt), flush=True)
        if verbose:
            prt()
        if cnt in skip_formulas:
            prt(end="(skipped)", flush=True)
        else:
            p = generate_polyhedra(t, bagnb, verbose=verbose, complex=complex)
            if p != None:
                p = PtsBag(p)
                p.name = "#" + str(bagnb)
                pts_bags.append(p)
                bagnb += 1
    total = mytime() - start
    if not verbose:
        prt()
    prt("Creation time: {:.3f} sec".format(total), flush=True)
    return pts_bags


def var_used_bool(b):
    """
    Measure which variables are used at all in a bag of polyhedra.

    >>> p = phwrap(eqns=[(0,0,0,4,0,6)], ieqs=[(0,0,2,5,0,4),(0,0,5,2,7,0),(0,0,0,2,0,5)])
    >>> var_used_bool([p])
    [False, True, True, True, True]
    """
    return [bool(i) for i in var_used_cnt(b)]


def var_used_cnt(b):
    """
    Count the number of uses of a specific variable in a bag of polyhedra.

    >>> p = phwrap(eqns=[(0,0,0,2,0,3)], ieqs=[(0,0,0,-1,0,0), (0,0,6,7,0,0), (0,0,5,2,7,0)])
    >>> var_used_cnt([p])
    [0, 2, 4, 1, 1]
    """
    uses = []
    for p in b:
        for h in p.Hrep():
            v = h.coeffs()     # this leaves out the absolute value
            if len(uses) < len(v):
                uses += [0] * (len(v) - len(uses))          # zero pad
            for cnt,coord in enumerate(v):
                uses[cnt] += coord != 0
    return uses


def likeness_15(b1, b2):
    """
    Predict size of intersection bag by v4 machine learning data.
    """
    l1, l2 = len(b1), len(b2)
    if l1 == 0 or l2 == 0:
        return inf
    u1 = var_used_bool(b1)
    #if not u1:
    #    return -inf
    #u2 = var_used_bool(b2)
    dim = len(u1)
    #uand = [i+j > 1 for i,j in zip(u1, u2)]
    #uor = [i+j > 0 for i,j in zip(u1, u2)]
    a = [
        log(l1,l1*l2),                                  #  3: b1: # polyhedra in bag
        avg_dim(b1) / dim,                              #  5: b1: avg object dim
        log(l2,l1*l2),                                  #  8: b2: # polyhedra in bag
        avg_dim(b2) / dim,                              # 10: b2: avg object dim
    ]
    pred = clf.predict([a])
    p = min(max(0, pred[0]), 1)  # safety
    return -p * l1 * l2


def likeness(b1, b2, like):
    """
    Calculate a measure of like-ness of two polyhedron bags.
    """
    if like == 15:
        return likeness_15(b1, b2)
    elif like == 14:
        return -len(b1)
    assert 0


def avg_dim(b):
    """
    Return the average dimension of a list of polyhedra.
    """
    return sum([p.dim() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_planes(b):
    """
    Return the average number of defining hyperplanes of a list of polyhedra.
    """
    return sum([p.n_Hrep() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_vrep(b):
    """
    Return the average number of defining vertices of a list of polyhedra.
    """
    return sum([p.n_Vrep() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_vertices(b):
    """
    Return the average number of vertices of a list of polyhedra.
    """
    return sum([p.n_vertices() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_rays(b):
    """
    Return the average number of rays of a list of polyhedra.
    """
    return sum([p.n_rays() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_lines(b):
    """
    Return the average number of lines of a list of polyhedra.
    """
    return sum([p.n_lines() for p in b]) / float(len(b)) if len(b) > 0 else 0

def avg_compact(b):
    """
    Return the average number of is_compact() of a list of polyhedra.
    """
    return sum([p.is_compact() for p in b]) / float(len(b)) if len(b) > 0 else 0


include_tests = 0
intersections = 0

class NoSolution(Exception):
    pass

class OutOfTime(Exception):
    pass


def insert_include(l, n):
    """
    Insert polyhedron n into list l, but check that n is not included
    in any of l's members and that n doesn't include any of l's members.
    If so, use the larger one.
    If both are equal, keep the one with the smaller combo index.
    """
    #return l + [n]                                     # to switch off inclusion test
    global include_tests
    strict_order = False
    # we're assuming that it is more likely that the new polyhedron is already included in the list,
    # so we check for that first.
    for idx,i in enumerate(l):
        # is n contained in i, i.e. something we already know?
        if i.contains(n):
            if strict_order and n.contains(i):
                # both are included in one another, i.e. they are equal.  keep the one with the lower combo index
                if combo_cmp(i.combo, n.combo) < 0:
                    l.insert(0, l.pop(idx))     # move to old i front of list
                else:
                    l.pop(idx)                          # remove old i with larger combo index
                    l.insert(0, n)                      # put new n to front of list
            else:
                # i is strictly larger than n
                l.insert(0, l.pop(idx))                 # move to front of list
            include_tests += 1 + int(strict_order)
            return l                                    # no need to continue
        include_tests += 1
    # n is not included in any of l's polyhedra, so it has to be added to the list anyway.
    # see if n includes any of the already existing ones.
    l2 = []
    for i in l:
        # is i contained in n?  if so, don't copy it to new list.
        if not n.contains(i):
            # here, all inclusions are strict, otherwise they would have been found in pass 1.
            l2.append(i)                                # append i to new list
        include_tests += 1
    l2.append(n)
    return l2


def myintersect(p, q):
    #t = mytime()
    r = p & q
    #t = mytime() - t
    #prt("peq={}\npie={}\nqeq={}\nqie={}\ntime={} hs={} vs={}\n".format(list(peq), list(pie), list(qeq), list(qie), t, r.n_Hrep(), r.n_Vrep()).replace(" ", ""), screen=False)
    r.combo = p.combo
    r.combo.update(q.combo)
    return r


def mk_stat_blob(b1, b2, l, empty_cnt, incl_cnt, drop_cnt):
    """
    Return statistics for both input and the output bag.

    They each contain:
        #polyhedra, #used variables
    """
    u1 = var_used_bool(b1)
    if not u1:
        return None
    u2 = var_used_bool(b2)
    mls = "v4: "
    for i in (b1, b2, l):
        mls += "{} {} {:.2f} {:.2f} {:.2f} ".format(len(i), sum(var_used_bool(i)), avg_planes(i), avg_dim(i), avg_compact(i))
    uand = [i+j > 1 for i,j in zip(u1, u2)]
    uor = [i+j > 0 for i,j in zip(u1, u2)]
    mls += "{} {} {} {} {} {}".format(len(u1), empty_cnt, incl_cnt, drop_cnt, sum(uand), sum(uor))
    return mls


def intersect_two(b1, b2, verbose=0, text="", like=9, endtime=inf, complete=False):
    """
    Intersect all elements of p with all elements of q.
    Only save non-empty results and make sure that no element
    of the resulting list is included in any other element of it.
    """
    if verbose:
        prt(end="[{}({}, {})]: {} * {} = {}".format(text, b1.name, b2.name, len(b1), len(b2), len(b1)*len(b2)), flush=True)
    # from the two bags, take all combinations of one polyhedron each and intersect those combinations.
    # if the intersection is empty, drop it.
    # if the intersection is contained in some other intersection we already have, drop it.
    # if the intersection is a superset of some other intersection we already have, drop that other intersection
    #    and add the new intersection.
    l = []                                              # new list of polyhedra
    empty_cnt = 0
    cnt = 0
    drop_cnt = 0
    tame = tame_it(2)
    sts = status_print()
    global include_tests
    global intersections
    if mytime() > endtime:
        raise OutOfTime
    for p in b1:
        for q in b2:
            cnt += 1
            if tame.go():
                if mytime() > endtime:
                    raise OutOfTime
                sts.print("({})".format(cnt))
            r = myintersect(p, q)                       # intersect
            intersections += 1
            if r.is_empty():                            # ignore if the intersection is empty
                empty_cnt += 1
                continue
            if complete and r.dim() != p.dim() - (q.space_dim() - q.dim()):
                drop_cnt += 1
                continue
            l = insert_include(l, r)
    sts.print("")
    incl_cnt = len(b1)*len(b2) - len(l) - empty_cnt - drop_cnt
    if verbose:
        lk = likeness(b1, b2, like)
        lstr = "" if like in (0,14) else " like={:.2f}%".format(lk * 100.0) if like in (10,12,13) else " like={:.1f}".format(float(lk))
        s = " => {} ({} empty, {} incl".format(len(l), empty_cnt, incl_cnt)
        if complete:
            s += ", {} drop".format(drop_cnt)
        s += "){}, dim={:.1f}, hs={:.1f}".format(lstr, avg_dim(l), avg_planes(l))  # "used={}, ", sum(var_used_bool(b2))
        if phwrap().has_v:
            s += ", vs={:.1f}".format(avg_vrep(l))
        prt(s, flush=True, flushfile=True)
    if len(l) == 0:
        raise NoSolution
    return PtsBag(l), mk_stat_blob(b1, b2, l, empty_cnt, incl_cnt, drop_cnt)


def highest_likeness(bags, like):
    minlike = []
    lmax = inf if like in minlike else -inf
    for i,b in enumerate(bags):
        for j,c in enumerate(bags):
            if like == 15 and j > 0:
                break
            if j >= i:
                break
            lk = likeness(b, c, like)
            if like in minlike:
                #lk = max(0.0, lk)
                #lk = min(1.0, lk)
                #prt("  {} {} {} {} {} {}".format(b.name, c.name, len(b), len(c), lk, lk * len(b) * len(c)))
                cmp = (1.0 - lk) * len(b) * len(c) < lmax
            else:
                cmp = lk > lmax
            if cmp:
                lmax = lk
                maxij = i, j
    j = maxij[0]
    i = maxij[1]
    assert j > i
    bj = bags[j]
    bags.pop(j)
    bi = bags[i]
    bags.pop(i)
    bags.insert(0, bj)
    bags.insert(1, bi)


# eqns - list of equalities. An entry equal to [1,7,3,4] represents the equality 1+7x1+3x2+4x3=0
# ieqs - list of inequalities. An entry equal to [1,7,3,4] represents the inequality 1+7x1+3x2+4x3>=0
def mkdict(l, is_eq, add_neg=False):
    """
    Convert a list of vectors (lists) to a dict (the 0-th coeff as value, the rest as key),
    optionally adding the negatives of those vectors as well since x = 2 implies x >= 2 as well as x <= 2).

    >>> sorted(mkdict([(1,2,3),(-4,-5,-6)], True).items())
    [((-5, -6), -4), ((2, 3), 1)]
    >>> sorted(mkdict([(1,2,3),(-4,-5,-6)], True, True).items())
    [((-5, -6), -4), ((-2, -3), -1), ((2, 3), 1), ((5, 6), 4)]
    """
    d = {}
    for v in l:
        v0 = v[0]                                       # the inhomogenous term
        vx = v[1:]                                      # the normal vector
        if is_eq:
            assert vx not in d or d[vx] == v0
            d[vx] = v0
        else:
            # if we have x-2 >= 0 (x >= 2) and x-4 >= 0 (x >= 4), then x-4 >= 0 (x >= 4) is true for both
            try:
                d[vx] = max(v0, d[vx])
            except KeyError:
                d[vx] = v0
        if add_neg:
            # negate, i.e. from 7x1+5x2 >= 2 generate 7x1+5x2 <= 2 as well
            vx = tuple([-i for i in vx])
            assert vx not in d or d[vx] == -v0          # add_neg must only be used for is_eq
            d[vx] = -v0
    return d


def dict_to_set(d):
    """
    Convert the normal vector/absolute dicts back to sets.

    >>> sorted(dict_to_set({(2,3):1, (5,6):4}))
    [(1, 2, 3), (4, 5, 6)]
    >>> sorted(dict_to_set({}))
    []
    """
    s = set()
    for k,v in d.items():
        s.add((v,) + k)
    return s


def join_dict(a, b):
    """
    >>> sorted(join_dict({1:2, 2:4}, {3:5, 4:9}).items())
    [(1, 2), (2, 4), (3, 5), (4, 9)]
    """
    d = a.copy()
    d.update(b)
    return d


def T_common_restrictions(b):
    """
    test companion function to cover directory randomness and version agnosticity.
    """
    r = common_restrictions(b)
    return [sorted([to_int(i) for i in s]) for s in r]

def common_restrictions(b):
    """
    Find the common restrictions for a list of polyhedra.

    >>> p = phwrap(eqns=[(1,2,3)], ieqs=[(4,5,6)])
    >>> q = phwrap(eqns=[(1,2,2)], ieqs=[(4,5,7)])
    >>> T_common_restrictions([p, q])
    [[], []]

    >>> p = phwrap(eqns=[], ieqs=[(4,5,6)])
    >>> q = phwrap(eqns=[], ieqs=[(4,5,6)])
    >>> T_common_restrictions([p, q])
    [[], [(4, 5, 6)]]

    >>> p = phwrap(eqns=[(1,2,3)], ieqs=[])
    >>> q = phwrap(eqns=[(2,2,3)], ieqs=[])
    >>> T_common_restrictions([p, q])
    [[], [(-1, -2, -3), (2, 2, 3)]]

    >>> p = phwrap(eqns=[], ieqs=[(1,2,3)])
    >>> q = phwrap(eqns=[], ieqs=[(2,2,3)])
    >>> T_common_restrictions([p, q])
    [[], [(2, 2, 3)]]
    """
    assert len(b) >= 2
    # initialize with first element of b
    comeq = mkdict(b[0].eqns_set, True)
    comie = join_dict(mkdict(b[0].ieqs_set, False), mkdict(b[0].eqns_set, True, True))

    # loop over the rest
    for p in b[1:]:
        eq = mkdict(p.eqns_set, True)
        ie = join_dict(mkdict(p.ieqs_set, False), mkdict(p.eqns_set, True, True))
        # check still common inequalities
        for k,v in comie.copy().items():
            if k not in ie:
                del comie[k]                            # it's not in both, i.e. delete in comie
            elif ie[k] != v:
                comie[k] = max(v, ie[k])               # if comie says x-2 >= 0 (x >= 2) and ie says x-4 >= 0 (x >= 4), then x-4 >= 0 (x >= 4) is true for both
        # check still common equalities
        for k,v in comeq.copy().items():
            if k not in eq:
                del comeq[k]                            # it's not in both, i.e. delete in comeq
            elif eq[k] != v:
                assert comie[k] >= v                    # this results from the adding of equations to comie
                comie[k] = max(comie[k], eq[k])         # if comeq says x-2 = 0 (x = 2) and eq says x-4 = 0 (x = 4), then x-2 >= 0 (x >= 2) is true for both
                del comeq[k]                            # no longer an equation
    # convert back
    return dict_to_set(comeq), dict_to_set(comie)


def mkset(l, add_neg=False):
    """
    Convert a list of vectors (lists) to a set,
    optionally adding the negatives of those vectors as well.

    >>> sorted(mkset([(1,2,3),(-4,-5,-6)]))
    [(-4, -5, -6), (1, 2, 3)]
    >>> sorted(mkset([(1,2,3),(-4,-5,-6)], True))
    [(-4, -5, -6), (-1, -2, -3), (1, 2, 3), (4, 5, 6)]
    """
    s = set()
    for v in l:
        s.add(v)
        if add_neg:
            s.add(tuple([-i for i in v]))
    return s


common_plane_time = 0
hull_time = 0

import random
def common_planes(pts_bags, old_common, verbose=False, convexhull=True, endtime=inf, complete=False):
    start = mytime()
    global common_plane_time, hull_time
    #prt("Finding common planes... ")
    combi = prod([len(i) for i in pts_bags])
    old_len = len(pts_bags)
    global intersections
    oeq, oie, oldc = old_common
    worked = False
    loop = True
    debug_hull = False
    max_planes = 5000
    slow_build = False

    # loop will run as long as we find new constraints
    while loop:
        if mytime() > endtime:
            raise OutOfTime
        common_eq = set()
        common_ie = set()
        hull_eq = set()
        hull_ie = set()
        one_eq = set()
        one_ie = set()
        one_cnt = 0
        one_combo = {}
        for cnt,b in enumerate(pts_bags):
            # make sure b.hull exists
            if "hull" not in b.__dict__:
                b.hull = False
            if len(b) == 1:
                # bags with just one polyhedron
                one_eq.update(mkset(b[0].equalities_tuple_list()))
                one_ie.update(mkset(b[0].inequalities_tuple_list()))
                one_cnt += 1
                one_combo.update(b[0].combo)
            elif b:
                # find common (in)equalities between all polyhedra.
                beq, bie = common_restrictions(b)
                if convexhull:
                    vertices = set()
                    rays = set()
                    lines = set()
                    for p in b:
                        vertices.update(mkset(p.vertices_tuple_list()))
                        rays.update(mkset(p.rays_tuple_list()))
                        lines.update(mkset(p.lines_tuple_list()))
                common_eq.update(beq)
                common_ie.update(bie)
                # build convex hull of all polyhedra of this bag
                b.hull = False
                if convexhull and len(vertices)+len(rays)+len(lines) <= max(avg_dim(b) * chull_f1, avg_planes(b) * chull_f3):
                    if debug_hull:
                        prt(end="[hull {}v".format(len(vertices)+len(rays)+len(lines)), flush=True)
                    start_hull = mytime()
                    hull = phwrap(vertices=vertices, rays=rays, lines=lines)
                    hull_time += mytime() - start_hull
                    assert not hull.is_empty()
                    hull_eq.update(mkset(hull.equalities_tuple_list()))
                    hull_ie.update(mkset(hull.inequalities_tuple_list()))
                    b.hull = True
                    if debug_hull:
                        prt(end=" {}h]".format(hull.n_Hrep()), flush=True)

        # remove already known planes, but not from one_*, since they are filtered out later
        common_eq -= oeq
        hull_eq -= oeq
        common_ie -= oie
        hull_ie -= oie

        # make sure we're not using too many planes
        # we're only limiting the number of inequalities.
        n = max_planes
        if convexhull:
            n = min(n, int(avg_dim(b) * chull_f2))
        xie = common_ie | hull_ie
        if len(xie) > n:
            if debug_hull:
                prt(end="[sample {} {}]".format(n, len(xie)), flush=True)
            xie = set(random.sample(xie, n))
        # any new planes?
        eq = one_eq | common_eq | hull_eq
        ie = one_ie | xie
        if not eq - oeq and not ie - oie:
            break
        oeq |= eq
        oie |= ie
        # build a polyhedron of all planes
        if debug_hull:
            prt(end="[cut {}h {} {} {} {} {} {}".format(len(eq)+len(ie), len(one_eq), len(common_eq), len(hull_eq), len(one_ie), len(common_ie), len(hull_ie)), flush=True)
            # prt("eq={}\nie={}".format(eq, ie))
        assert eq | ie
        if slow_build:
            c = None
            for cnt,i in enumerate(eq):
                s = "".join([(" {:+}*x{}".format(j, c2) if j else "") for c2,j in enumerate(i[1:])])
                s = "{}{} == 0".format(i[0], s)
                prt("Adding equality {}/{}: {}".format(cnt, len(eq), s), flush=True, flushfile=True)
                c1 = phwrap(eqns=[i])
                if c is None:
                    c = c1
                else:
                    c &= c1
            for cnt,i in enumerate(ie):
                s = "".join([(" {:+}*x{}".format(j, c2) if j else "") for c2,j in enumerate(i[1:])])
                s = "{}{} >= 0".format(i[0], s)
                prt("Adding inequality {}/{}: {}".format(cnt, len(ie), s), flush=True, flushfile=True)
                c1 = phwrap(ieqs=[i])
                if c is None:
                    c = c1
                else:
                    c &= c1
        else:
            c = phwrap(eqns=eq, ieqs=ie)
        if debug_hull:
            if phwrap().has_v:
                prt(end=" {}v]".format(c.n_Vrep()), flush=True)
            else:
                prt(end="]", flush=True)
        if c.is_empty():
            # empty polyhedron, i.e. no solution
            common_plane_time += mytime() - start
            raise NoSolution
        if convexhull:
            hull_ok = True
            # test against polyhedron without hull
            if debug_hull:
                prt(end="[recut {}h".format(len(eq)+len(ie)), flush=True)
            c2 = phwrap(eqns=eq, ieqs=common_ie | one_ie)
            if debug_hull:
                prt(end=" {}v]".format(c.n_Vrep()), flush=True)
            if c.n_Vrep() > c2.n_Vrep() * chull_f4:
                hull_ok = False
                c = c2
                if debug_hull:
                    prt(end="[not used]", flush=True)
        c.combo = one_combo
        # is new c any more restrictive?
        if oldc is not None and c.contains(oldc):
            break
        oldc = c

        loop = False
        worked = True
        bags = []
        cnt = 1
        #verbose = True
        prt(" Applying common planes (codim={}, hs={}{})...".format(c.space_dim() - c.dim(), c.n_Hrep(), ", vs={}".format(c.n_Vrep()) if phwrap().has_v else ""), flush=True)
        for b in pts_bags:
            empty_cnt = 0
            if len(b) != 1:
                prt(end=" [{}/{} ({})]: {}".format(cnt, len(pts_bags) - one_cnt, b.name, len(b)), flush=True, screen=verbose)
                cnt += 1
                b2 = []
                pcnt = 0
                tame = tame_it(2)
                sts = status_print()
                for p in b:
                    pcnt += 1
                    if verbose and tame.go():
                        if mytime() > endtime:
                            raise OutOfTime
                        sts.print("({})".format(pcnt))
                    # strangely, it's faster to use r = p & c compare to r &= c (17.5s vs. 18.1s on bluthgen0)
                    r = myintersect(p, c)
                    intersections += 1
                    if r.is_empty():
                        empty_cnt += 1
                        continue
                    b2 = insert_include(b2, r)
                if verbose:
                    sts.print("")
                b2 = PtsBag(b2)
                b2.name = b.name
                bags.append(b2)
                loop |= len(b2) == 1                    # loop if resulting bag has only one polyhedron
                incl_cnt = len(b) - len(b2) - empty_cnt
                prt(" => {} ({} empty, {} incl), dim={:.1f}, hs={:.1f}{}{}".format(len(b2), empty_cnt, incl_cnt,  # used={}, sum(var_used_bool(b2))
                    avg_dim(b2), avg_planes(b2),
                    ", vs={:.1f}".format(avg_vrep(b2)) if phwrap().has_v else "",
                    " (hull)" if b.hull and hull_ok else ""), flush=True, screen=verbose)
                if len(b2) == 0:
                    common_plane_time += mytime() - start
                    raise NoSolution
        pts_bags = bags
        if not pts_bags:
            break
    # filter bags with just one polyhedron.  their constraits have already been intersected with the other bags.
    # this might happen, since the intersection only starts if eq and ie and c contains something new.
    pts_bags = [p for p in pts_bags if len(p) != 1]
    # make sure one_combo is not lost for the same reasons
    if not pts_bags:
        # if no bags left, return the common intersection c
        pts_bags = [PtsBag([c])]
        pts_bags[0].name = "c"
    for p in pts_bags[0]:
        p.combo.update(one_combo)
    if worked:
        prd = prod([len(i) for i in pts_bags])
        combi = combi // prd if combi / prd >= 100 else combi / prd
        prt(" Savings: factor {:,}{}, {} formulas".format(combi,
         (" (10^{})".format(myround(log10(combi))) if combi > 10 else ""), old_len - len(pts_bags)), flush=True)
    common_plane_time += mytime() - start
    return pts_bags, (oeq, oie, oldc)


def intersect_all(pts_bags, verbose=0, algo=0, like=9, common=True, sorting=0, resort=False, convexhull=True, fname1=None,
        max_run_time=inf, bag_order=[], complete=False, nocons=False):
    """
    Calculate the intersections of one polyhedron per bag.

    Input: 'pts_bags' is a list of lists of polyhedra ("bags of polyhedra").

    Output: a list of polyhedra.

    Take all combinations of one polyhedron per "bag".
    Intersect the polyhedra of each combination.
    If, in the result, one polyhedron is included in another, drop the included one from the solution set.
    """
    stats = {"max_pt": 0, "maxin_pt": 0, "aborted": False}
    if not pts_bags:                                    # no input
        return [], stats
    combis = prod(len(i) for i in pts_bags)
    old_common = set(), set(), None
    dim = -2
    for i in pts_bags:
        if i:
            dim = i[0].space_dim()
            break
    if verbose:
        if dim != -2:
            prt("Dimension: {}".format(dim))
            prt("Formulas: {}".format(len(pts_bags)))
            prt(end="Order:")
            for i in pts_bags:
                prt(end=" {}".format(i.name[1:]))
            prt()
            prt(end="Possible combinations: ")
            for cnt,i in enumerate(pts_bags):
                prt(end="{}{} ".format("* " if cnt > 0 else "", len(i)))
            prt("= {:,}{}".format(combis, (" (10^{})".format(myround(log10(combis)))) if combis > 0 else ""), flush=True)
    if complete and len(pts_bags) > dim:
        prt("*** Warning: --complete used on overdetermined system!  Solution will be empty!")
    elif nocons and len(pts_bags) != dim:
        prt("*** Warning: number of equations and number of variables differ!")
    if fname1:
        fname1 += "-temp"
    global intersections, include_tests, common_plane_time, hull_time
    intersections = 0
    include_tests = 0
    common_plane_time = 0
    hull_time = 0
    start = mytime()
    ftime = 0
    endtime = mytime() + max_run_time

    try:
        if algo == 2:
            prt("Looking for best start bag:"),
            lmax = -1
            for i,b in enumerate(pts_bags):
                for j,c in enumerate(pts_bags):
                    if j > i:
                        lk = likeness(b, c, like)
                        if lk > lmax:
                            lmax = lk
                            maxij = (i, j)
            m = maxij[0]
            prt(pts_bags[m].name)
            pts_bags = [pts_bags[m]] + pts_bags[:m] + pts_bags[m+1:]
        run = 1
        runs = len(pts_bags) - 1
        work_cnt = 1
        while len(pts_bags) > 1:
            if common:
                # find common planes and apply them to all bags
                old = len(pts_bags)
                pts_bags, old_common = common_planes(pts_bags, old_common, verbose=verbose > 1, convexhull=convexhull and run == 1, endtime=endtime, complete=complete)
                if len(pts_bags) <= 1:
                    break
                runs -= old - len(pts_bags)
                # if resorting is on, keep it sorted in every step
                if resort or run == 1:
                    if sorting == 1:
                        pts_bags = sorted(pts_bags, key=lambda x: len(x))
                    elif sorting == 2:
                        pts_bags = sorted(pts_bags, key=lambda x: -len(x))

            if algo == 0:
                # standard breadth first
                l, mls = intersect_two(pts_bags[0], pts_bags[1], verbose, text="{}/{} ".format(run, runs), like=like, endtime=endtime)
                pts_bags = [l] + pts_bags[2:]
            elif algo == 1:
                # join and conquer
                i = 0
                bags2 = []
                while i < len(pts_bags) - 1:
                    b1 = pts_bags[i]
                    b2 = pts_bags[i+1]
                    l, mls = intersect_two(b1, b2, verbose, like=like, text="{}: ".format(run), endtime=endtime)
                    bags2.append(l)
                    i += 2
                # odd number of bags?
                if i < len(pts_bags):
                    bags2.append(pts_bags[i])
                pts_bags = bags2
            elif algo == 2:
                # find the next one that has the highest likeness
                lmax = -1
                nhmin = 2**63
                if like == 9 and True:
                    gavg = combis ** (1.0 / len(pts_bags))
                    if gavg > len(pts_bags[0]):
                        bags2 = [i for i in pts_bags[1:] if len(i) >= gavg]
                        bags1 = [i for i in pts_bags[1:] if not len(i) >= gavg]
                    else:
                        bags2 = [i for i in pts_bags[1:] if len(i) <= gavg]
                        bags1 = [i for i in pts_bags[1:] if not len(i) <= gavg]
                    #prt(gavg, [len(p) for p in bags2])
                else:
                    bags1 = []
                    bags2 = pts_bags[1:]
                for b in bags2[:]:
                    lk = likeness(pts_bags[0], b, like)
                    nh = avg_planes(b)
                    bags2 = bags2[1:]
                    if lk > lmax or (like == 3 and lk == lmax and nh < nhmin):
                        lmax = lk
                        nhmin = nh
                        b1max = bags1[:]
                        b2max = bags2[:]
                        bmax = b
                    bags1.append(b)
                l, mls = intersect_two(pts_bags[0], bmax, verbose, like=like, text="{}/{} ".format(run, runs), endtime=endtime)
                l.name = "w"
                combis = combis / (len(pts_bags[0]) * len(bmax)) * len(l)
                pts_bags = [l] + b1max + b2max
            elif algo == 3:
                # if a bag order was specified, the specified bags have already been sorted to the front.
                # if we encounter an unlisted bag it means the ordered bags list has been consumed and
                # the automatism should take over.
                if not int(pts_bags[1].name[1:]) in bag_order:
                    # find the next pair that has the highest likeness
                    highest_likeness(pts_bags, like)         # sort the two with the highest likeness to the front
                stats["maxin_pt"] = max(stats["maxin_pt"], len(pts_bags[0]) * len(pts_bags[1]))
                l, mls = intersect_two(pts_bags[0], pts_bags[1], verbose, like=like, text="{}/{} ".format(run, runs), endtime=endtime, complete=complete)
                l.name = "w{}".format(work_cnt)
                work_cnt += 1
                combis = combis / (len(pts_bags[0]) * len(pts_bags[1])) * len(l)
                pts_bags = [l] + pts_bags[2:]
            run += 1
            stats["max_pt"] = max(stats["max_pt"], len(pts_bags[0]))
            if fname1:
                save_polyhedra(pts_bags, fname1, quiet=True)
            if mls:
                print(mls, file=mls_file)
                mls_file.flush()
        assert len(pts_bags) == 1

    except NoSolution:
        pts_bags = [PtsBag([])]
    except OutOfTime:
        prt("\n****** Computation aborted!!!  Out of time!!! ******")
        stats["aborted"] = True
        pts_bags = [PtsBag([])]

    stats["max_pt"] = max(stats["max_pt"], len(pts_bags[0]))
    total = mytime() - start - ftime
    prt("Intersection time: {:.3f} sec, total intersections: {:,}, total inclusion tests: {:,}{}".format(
        total, intersections, include_tests, " (aborted)" if stats["aborted"] else ""), flush=True)
    if common:
        prt("Common plane time: {:.3f} sec{}".format(common_plane_time, ", convex hull time: {:.3f} sec".format(hull_time) if convexhull else ""), flush=True)
    if fname1:
        try:
            os.remove(fname1 + "-polyhedra.txt")
        except OSError:
            pass
    return pts_bags[0], stats


def status(time, file=sys.stdout):
    print("\n" + "-" * 70, file=file)
    if IS_SAGE:
        print(version(), file=file)
    else:
        print(end="Plain ", file=file)
    print("Python {}".format(sys.version), file=file)
    #import graph
    #import fuse
    print("This is ptcut v{}".format(vsn), file=file)
    print("It is now {}".format(time), file=file)
    print("command line: {}".format(" ".join(sys.argv)), file=file)


def T_get_ep(i):
    r = get_ep(i)
    return str(r[0]), r[1]

def get_ep(i):
    """
    Read the value of 1/ep.  Can be one of four formats:
    1. "<int>": ep = 1/<int>.
    2. "p<idx>": ep is the reciprocal of the smallest prime number with <idx> many digits.
    3. "c<idx>": ep is the reciprocal of the smallest prime number with <idx> many digits + 1 (thus it's surely composite).
    4. "<float>": ep 1/<float>, where <float> is acurate (it's expressed as a rational from string representation).

    >>> T_get_ep("5")
    ('1/5', '5')
    >>> T_get_ep("01001")
    ('1/1001', '1001')
    >>> T_get_ep("p3")
    ('1/1009', 'p3')
    >>> T_get_ep("c4")
    ('1/10008', 'c4')
    >>> T_get_ep("e2")
    ('1/100', 'e2')
    >>> T_get_ep("1.1")
    ('10/11', '11d10')
    """
    if i[0] in "pce" and i[1:].isdigit():
        idx = int(i[1:])
        if i[0] == "e":
            rep = 10 ** idx
        else:
            try:
                rep = make_prime(idx) + (1 if i[0] == "c" else 0)
            except IndexError:
                print("No prime with {} digits found".format(idx))
                sys.exit(1)
        i = i[0] + str(idx)                             # make it canonical
        ret = fract(1, rep), i
    else:
        try:
            rep = int(i)
            ret = fract(1, rep), str(rep)               # make it canonical
        except ValueError:
            from fractions import Fraction
            rep = Fraction(i)                           # construct from string to avoid rounding problems
            rep = fract(rep.numerator, rep.denominator) # convert to SAGE, possibly
            n = "{}d{}".format(rep.numer(), rep.denom())
            ret = 1 / rep, n
    assert 0 < ret[0] < 1
    return ret


default_epname = "5"

def main():
    total_time = mytime()
    from datetime import datetime
    boot_time = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    status(boot_time)

    print_out = False
    sorting = 1
    verbose = 0
    eps = []
    models = []
    algo = 3
    like = 14
    collect = 0
    skip_formula = []
    common = True
    sumup = False
    rnd = 0
    force = False
    keep_coeff = False
    resort = False
    paramwise = True
    test = False
    convexhull = True
    grid_data = []
    log_file_name = None
    complex = False
    save_solutions = True
    bag_order = []
    fusion = False
    save_solutions_to_log = False
    con_type = 0
    connected_components = False
    multiple_lpfiles = False
    jeff = False
    multi_log = False
    runs = 1
    max_run_time = inf
    log_append = False
    work_dir = ""
    db_dir = "db"
    complete = False
    nocons = False
    comp_sol = True
    lifting = True
    set_phwrap(PhWrapPplCPolyhedron)

    for i in sys.argv[1:]:
        if collect:
            if collect == 1:
                skip_formula.extend([int(j) for j in i.split(",")])
            elif collect == 2:
                global chull_f1
                chull_f1 = bestnum(i)
            elif collect == 3:
                global chull_f2
                chull_f2 = bestnum(i)
            elif collect == 4:
                for j in i.split(","):
                    eps.append(get_ep(j))
            elif collect == 5:
                global chull_f3
                chull_f3 = bestnum(i)
            elif collect == 6:
                global chull_f4
                chull_f4 = bestnum(i)
            elif collect == 7:
                grid_data.extend(read_grid_data(i))
            elif collect == 8:
                rnd = int(i)
            elif collect == 9:
                log_file_name = i
                multi_log = True
            elif collect == 10:
                bag_order.extend([int(j) for j in i.split(",")])
            elif collect == 11:
                con_type = int(i)
            elif collect == 12:
                runs = int(i)
            elif collect == 13:
                max_run_time = float(i)
            elif collect == 14:
                work_dir = i
            elif collect == 15:
                db_dir = i
            collect = 0
        elif i.startswith("-"):
            if i == "-t":
                runs = 2**63
            elif i == "-p":
                print_out = True
            elif i == "--sortup":
                sorting = 1
            elif i == "--sortdown":
                sorting = 2
            elif i == "--shuffle":
                sorting = 3
            elif i == "--noshuffle":
                sorting = 0
            elif i.startswith("-a"):
                algo = int(i[2:])
            elif i.startswith("-l"):
                like = int(i[2:])
            elif i == "--bp":
                set_phwrap(PhWrapPolyhedronPPL)
            elif i == "--bc":
                set_phwrap(PhWrapPplCPolyhedron)
            elif i == "--bd":
                set_phwrap(PhWrapPolyhedronCDD)
            elif i == "--bn":
                set_phwrap(PhWrapPolyhedronNormaliz)
            elif i == "--bf":
                set_phwrap(PhWrapPolyhedronField)
            elif i == "--bm":
                set_phwrap(PhWrapPolyhedronPolymake)
            elif i.startswith("-v"):
                for j in i[1:]:
                    if j != 'v':
                        break
                    verbose += 1
            elif i == "--verbose":
                verbose += 1
            elif i == "--simple":
                models.extend(biomd_simple)
            elif i == "--easy":
                models.extend(biomd_easy)
            elif i == "--fast":
                models.extend(biomd_fast)
            elif i == "--slow":
                models.extend(biomd_slow)
            elif i == "--slowhull":
                models.extend(biomd_slowhull)
            elif i == "--hard":
                models.extend(biomd_hard)
            elif i == "--all":
                models.extend(biomd_all)
            elif i == "--skip":
                collect = 1
            elif i == "--common":
                common = True
            elif i == "--nocommon" or i == "--nc":
                common = False
            elif i == "--sum" or i == "--sumup":
                sumup = True
            elif i == "--nosum" or i == "--nosumup":
                sumup = False
            elif i == "-C":
                sumup = True
                keep_coeff = True
                paramwise = False
            elif i == "-f":
                force = True
            elif i == "--keep-coeff":
                keep_coeff = True
            elif i == "--resort":
                resort = True
            elif i == "--merge-param" or i == "--merge-params":
                paramwise = False
            elif i == "--test":
                test = True
            elif i == "--nohull" or i == "--nh":
                convexhull = False
            elif i == "--hull":
                convexhull = True
            elif i == "--hull1":
                convexhull = True
                collect = 2
            elif i == "--hull2":
                convexhull = True
                collect = 3
            elif i == "--hull3":
                convexhull = True
                collect = 5
            elif i == "--hull4":
                convexhull = True
                collect = 6
            elif i == "--ep":
                collect = 4
            elif i.startswith("-e"):
                for j in i[2:].split(","):
                    eps.append(get_ep(j))
            elif i == "--np" or i == "--no-progress":
                global progress
                set_progress(False)
            elif i == "--grid":
                collect = 7
            elif i == "-r" or i == "--round":
                collect = 8
            elif i.startswith("-r"):
                rnd = int(i[2:])
            elif i == "--log":
                collect = 9
            elif i == "--append":
                log_append = True
            elif i == "--complex" or i == "-c":
                complex = True
            elif i == "--nosave":
                save_solutions = False
            elif i == "--order":
                collect = 10
            elif i == "--fusion":
                fusion = True
            elif i == "--nofusion":
                fusion = False
            elif i == "--soltolog" or i == "--stl":
                save_solutions_to_log = True
            elif i == "--contype":
                collect = 11
            elif i == "--concomp" or i == "--cc":
                connected_components = True
            elif i == "--multiple-lpfiles":
                multiple_lpfiles = True
            elif i == "--jeff":
                jeff = True
            elif i == "--runs":
                collect = 12
            elif i == "--maxruntime":
                collect = 13
            elif i == "-q":
                verbose = 0
            elif i == "--dir":
                collect = 14
            elif i == "--db":
                collect = 15
            elif i == "--complete":
                complete = True
            elif i == "--nocons":
                nocons = True
            elif i == "--nocomp":
                comp_sol = False
            elif i == "--nolift":
                lifting = False
        else:
            models.append(i)

    time_prec()

    if not models:
        print("Please specify a model")
        sys.exit(1)

    if sumup and not keep_coeff:
        print("--sumup requires --keep-coeff !")
        sys.exit(1)

    if complete:
        common = False
        nocons = True

    if not common:
        convexhull = False

    if not phwrap().has_v:
        convexhull = False

    if not eps:
        eps.append((fract(1, int(default_epname)), default_epname))

    if jeff:
        complex = True
        grid_data = []
        sumup = False
        keep_coeff = False
        paramwise = True
        eps = [(0, "x")]
        rnd = 0      # dummy

    if grid_data:
        save_solutions = False
        print_out = False

    if like == 15:
        import pickle
        global clf
        clf = pickle.load(open("mlmodel4.sav", "rb"))

    global mls_file
    mls_file = open("mlstats.txt", "a")

    total_runs = 0
    first_run = True
    mismatch = 0
    while runs > 0:
        for ep, epname in eps:
            for mod in models:
                flags = ("s" if sumup else "") + ("k" if keep_coeff else "") + ("" if paramwise else "m") + ("n" if nocons else "")
                pflags = flags + ("c" if complex else "")
                sflags = pflags + ("f" if fusion else "") + ("p" if complete else "")
                mod_dir = (work_dir + os.sep if work_dir else "") + "{}{}{}{}".format(db_dir, os.sep, mod, os.sep)
                fname0 = "{}ep{}{}".format(mod_dir, epname, ("-r{}".format(rnd) if rnd > 0 else ""))
                tfname1 = fname0 + ("-" + flags if flags else "")
                fname1 = fname0 + ("-" + pflags if pflags else "")
                sfname1 = fname0 + ("-" + sflags if sflags else "")

                if not multi_log:
                    # each model has its own log file
                    prt.close_log_file()
                    log_file_name = sfname1 + "-log.txt"

                assert log_file_name
                if not prt.get_log_file():
                    # log file set (through --log as multi_log or individually), but not yet opened
                    prt.set_log_file(open(log_file_name, "a" if log_append else "w"))
                    if prt.get_log_file():
                        status(boot_time, file=prt.get_log_file())

                prt()
                prt("-" * 70)
                if ep == 0:
                    epstr = "x"
                elif 1 / ep < 11000:
                    epstr = str(ep)
                else:
                    lg = int(floor(log10(1/ep)))
                    add = 1/ep - 10**lg
                    epstr = "1/(10**{}+{})".format(lg, add) if add else "1/10**{}".format(lg)
                tropstr = "" if jeff else "ep={}, round={}, complex={}, nocons={}, sumup={}, keep_coeff={}, paramwise={}, ".format(
                epstr, rnd, complex, nocons, sumup, keep_coeff, paramwise)
                prt("Solving {} with {}complete={}, fusion={}, algorithm {}, resort={}, common={}, convexhull={}, likeness {}, backend {}".format(
                    mod, tropstr, complete, fusion, algo, resort, common, (convexhull, chull_f1, chull_f2, chull_f3, chull_f4), like, phwrap().name))
                fep = float(1/ep) ** (1/10**rnd)
                prt("Effective epsilon: 1/{}{}".format(fep, ", 1/(1 + {:.3g})".format(fep - 1) if fep <= 1.1 else ""))
                prt()

                trop_cache = None
                ts = None
                for override in sample_grid(grid_data):
                    # if grid sampling, print which parameters we're using
                    if grid_data:
                        param_str = ", ".join(["{} = {}".format(k, v) for k,v in override])
                        prt("\nGrid point: {}".format(param_str))

                    # load or compute tropical system
                    same_trop = False
                    if jeff:
                        ts = load_jeff_system(mod_dir)
                    else:
                        old_ts = ts
                        ts = None if force or grid_data else load_tropical(tfname1)
                        if not ts:
                            prt(end="Computing tropical system... ", flush=True)
                            ts, trop_cache = tropicalize_system(mod, mod_dir, ep=float(ep), scale=10**rnd, sumup=sumup, verbose=verbose, keep_coeff=keep_coeff, paramwise=paramwise, param_override=dict(override), cache=trop_cache, nocons=nocons)
                            if not grid_data:
                                save_tropical(ts, tfname1)
                            same_trop = ts == old_ts    # set to True if grid sampling and nothing changed
                    if same_trop:
                        prt("Skipping calculation, since nothing changed.")
                        rs = old_rs
                        stats = old_stats
                        sol = None
                    else:
                        # load solution, if existant
                        sol = None
                        if comp_sol and not complex and not nocons and not fusion and not grid_data and ep == fract(1, 5) and rnd == 0 and mod.startswith("BIOMD"):
                            try:
                                n = int(mod[5:])
                            except ValueError:
                                pass
                            else:
                                sol = load_satyas_solution(n)

                        pts_bags = None if force or grid_data else load_polyhedra(fname1)
                        if not pts_bags:
                            prt(end="Creating polyhedra... ", flush=True)
                            pts_bags = make_polyhedra(ts, skip_formula, verbose=verbose, complex=complex)
                            if not grid_data:
                                save_polyhedra(pts_bags, fname1)
                        prt()

                        if sorting == 1:
                            prt("Sorting ascending...")
                            pts_bags = sorted(pts_bags, key=lambda x: len(x))
                        elif sorting == 2:
                            prt("Sorting descending...")
                            pts_bags = sorted(pts_bags, key=lambda x: -len(x))
                        elif sorting == 3:
                            prt("Shuffling...")
                            shuffle(pts_bags)

                        # if --order was specified, order listed bags to the beginning
                        if bag_order:
                            bags = []
                            for i in bag_order:
                                for j in range(len(pts_bags)):
                                    if not pts_bags[j] is None and pts_bags[j].name == "#{}".format(i):
                                        bags.append(pts_bags[j])
                                        pts_bags[j] = None
                            for b in pts_bags:
                                if not b is None:
                                    bags.append(b)
                            pts_bags = bags
                            #sorting = 0

                        rs, stats = intersect_all(pts_bags, verbose=1, algo=algo, like=like, common=common, sorting=sorting,
                            resort=resort, convexhull=convexhull, fname1=fname1, max_run_time=max_run_time, bag_order=bag_order,
                            complete=complete, nocons=nocons)
                        old_rs, old_stats = rs, stats

                    if not stats["aborted"]:
                        #if fusion and not same_trop:
                        #    # fuse polyhedra
                        #    from fuse import fuse_polyhedra
                        #    ftime = mytime()
                        #    olen = len(rs)
                        #    fuse_polyhedra(rs, progress=True)
                        #    ftime = mytime() - ftime
                        #    if len(rs) != olen:
                        #        prt("Fusion reduced polyhedra from {} to {}.  Fusion time: {:.3f} sec".format(olen, len(rs), ftime))

                        # print number of solutions
                        s = "Solutions: {}".format(len(rs))
                        if rs:
                            s += ", dim={:.1f}, hs={:.1f}{}".format(avg_dim(rs), avg_planes(rs),
                            ", vs={:.1f}".format(avg_vrep(rs)) if phwrap().has_v else "")
                        s += ", max={}, maxin={}".format(stats["max_pt"], stats["maxin_pt"])
                        prt(s)
                        import collections
                        fvec = collections.Counter([p.dim() for p in rs])
                        if fvec:
                            prt("f-vector: {}".format([fvec.get(i, 0) for i in range(max(fvec.keys())+1)]))

                        # sort into a distict representation
                        if save_solutions or save_solutions_to_log:
                            prt("Canonicalizing...", flush=True)
                            rs = canonicalize(rs)

                        # count connected components
                        #if connected_components:
                        #    adj, adj_str = build_graph(rs, con_type=con_type, dbg=True)
                        #    import graph
                        #    cc = graph.connected_components(adj)
                        #    prt("Connected components: {}".format(cc))
                        #    prt("Graph ID: {}".format(graph.graph_name(adj_str)))

                        prt.flush_log_file()

                        if (save_solutions or save_solutions_to_log) and first_run:
                            prt(end="Saving solutions... ", flush=True)
                            dim = len(rs[0].Hrep()[0].vector()) - 1 if rs else -1
                            vars = ["x{}".format(i+1) for i in range(dim)]
                            if save_solutions and not multiple_lpfiles or save_solutions_to_log:
                                s = sol_to_string(rs, vars, ep=ep if lifting else None, scale=10**rnd)
                            if save_solutions:
                                if multiple_lpfiles:
                                    sol_to_lpfile(rs, fname1, vars, ep=ep if lifting else None, scale=10**rnd)
                                else:
                                    sol_string_to_one_lpfile(s, sfname1)
                            if save_solutions_to_log:
                                prt(end=s, screen=False)
                            prt(flush=True)

                        if sol:
                            notfound = compare_solutions(rs, sol)
                            if notfound is not None:
                                if notfound == 0:
                                    prt("Solutions match Satya's solutions")
                                else:
                                    prt("Solutions have {} differences from Satya's solutions".format(notfound))
                                    mismatch += 1

                        if print_out:
                            prt()
                            prt("Solution:")
                            if not rs:
                                prt("Solution set is empty")
                            #for i in rs:
                            #    prt(i.Hrep())
                            for i in rs:
                                prt("    {}".format(i.Vrep()))
                            if sol:
                                prt()
                                prt("Satya's solution is:")
                                for i in sol:
                                    prt("    {}".format(i.Vrep()))

                        prt("Calculation done.")
                    total_runs += 1

        runs -= 1
        first_run = False

    total_time = mytime() - total_time
    if total_runs > 1:
        prt("Total time: {:.3f} sec for {} runs, avg. {:.3f} sec".format(total_time, total_runs, total_time/total_runs), flush=True)
        prt("Mismatches: {}".format(mismatch))


if __name__ == "__main__":
    import doctest
    import phwrapper
    doctest.testmod(phwrapper, verbose=False)
    import biomd
    doctest.testmod(biomd, verbose=False)
    import util
    doctest.testmod(util, verbose=False)
    #import fuse
    #doctest.testmod(fuse, verbose=False)
    #import graph
    #doctest.testmod(graph, verbose=False)
    doctest.testmod(verbose=False)

    main()
